#INCLUDE P16F628A.inc
;
; Lock a 10MHz OCXO to a GPS 1ppS (1 pulse per second)
; CPU clocked from OCXO. TMR2 counts to 250 instruction cycles (1000 clock
; cycles) and provides a PWM signal (dithered in software) to provide the OCXO
; control voltage. Software counts 10,000 TMR2 interrupts to measure 1 second.
; Initially TMR2 is adjusted every 1ppS while varying the control voltage to
; bring the OCXO into lock. Once locked the OCXO frequency is adjusted to keep
; TMR2 and the 1ppS aligned.
;
    __CONFIG    _EXTCLK_OSC & _WDT_OFF & _PWRTE_ON & _LVP_OFF
;
; Exernal Clock (10MHz)
; WDT (Watch Dog Timer) disabled
; PWRT (Power on Timer) enabled
; Low Voltage Programming disabled :-
; RB4/PGM pin has digital I/O function, HV on MCLR must be used for programming
;
; links to arithmetic module
	extern Mshift
	extern MpushLit
	extern Mpush2U
	extern Mpush2
	extern Mpush3U
	extern Mpush3
	extern Mpush4
	extern Mpop4
	extern Mpop3
	extern Madd
	extern Msubtract
	extern Multiply
	extern MidMul
	extern MidDiv
	extern Mdivide
    ;    extern round
; links to transmit module
	extern TXinit
	extern INTtx
	extern TXhex
	extern TXtest
; entry from transmit 
	global INTunknown
	global EndInt
; Limiting test length to max 22.5 min - 1350 samples
; THIS VALUE DETERMINED BY EXPERIMENTATION
; See Section FinaLoop for how the test period is calculated
#define TestLimit 0x05
; set the desired loop value when the 1ppS (RB0) interrupt occurs.
; FFFE is three TMR2 interrupts before 'end of second'
#define	RB0Loop 0xFFFD
RB0Lo	EQU	RB0Loop & 0xFF
RB0Hi	EQU	RB0Loop >> 8
; the desired TMR2 value captured by the RB0 interrupt.
; Ideally the 1ppS arrives half way between the end of a TMR2 interrupt and
; the start of the next one. At the end of the TMR2 interrupt, TMR2 will be
; about 50 and is reset at 250 so the desired arrival time of the 1ppS is
; 150. It takes the RB0 interrupt about 16 instruction times before it captures
; the timer so the desired captured value is 166.
TargVal	EQU	0xA6
;
; Multibyte numbers are stored little endian
;
; Pins on processor
;
#define	    LED1    PORTA,RA0 ; output port for LED 1
#define	    LED2    PORTA,RA1 ; output port for LED 2
DisCharge   EQU     RA2 ; Voltage for phase detector
PreCharge   EQU     RA3 ; Thermistor
; unused    EQU     RA4
; Reset	    EQU     RA5 ; reset - not referenced by program - input
#define	    MakePPS PORTA,RA6 ; output port to make our 1pps
; ClockIn   EQU     RA7 ; Clock input - not referenced by program
; IntLine   EQU     RB0 ; Detector input - causes interrupt
;                        - not referenced by program
; UARTrx    EQU     RB1 ; Serial in - to UART - not referenced by program
; UARTtx    EQU     RB2 ; Serial out - from UART - not referenced by program
; PWMout    EQU     RB3 ; PWM output - not referenced by program
#define	    ClrPPSou    PORTB,RB4 ; output port to disable 1pps output
#define	    ClrPPSin    PORTB,RB5 ; output port to disable 1pps input from GPS unit
; unused    EQU     RB6 ; reserved for In Circuit Programming
; unused    EQU     RB7 ; reserved for In Circuit Programming

; #### data memory locations ####

; -------------- common --------------
      udata_shr   

; areas for saving state during interrupts
;
SaveW       RES     1   ; save W for interrupt
SaveS       RES     1   ; save status for interrupt
;
; run variables
;
Mtemp       RES     4   ; Mainline work area
;
TstLen      RES     1   ; averaging period = int(675*2^TstLen-3) seconds
TstLoop     RES     3   ; General purpose counter
; if there is a failure, the value in ErrVal is flashed on the LEDs
ErrVal      RES     1   ; an error value

; -------------- bank 0 --------------
bnk0      udata       0x20

LoopC       RES     2   ; number of times through TMR2 loop,
                            ; reset each second
Ticks       RES     3   ; TMR2+LoopC when RB0 interrupt is taken
Tcomp       RES     1   ; comparison result

Flags	    RES     1   ; state flags mainly for interrupt communication
; use these bits in Flags
#define	    LoopEnd     Flags,0   ; bit set once a second by TMR2 routine
#define	    GPStick     Flags,1   ; bit set if GPS 1ppS (RB0) detected
#define	    ResetCk     Flags,2   ; bit set - counters reset on RB0
#define	    VRon	Flags,3	; set when using 2V from the Voltage Reference
#define	    Timer	Flags,4	; ** DEBUG trace timer
#define	    ValidGPS	Flags,5	; ValidGPS
#define	    SmallAdj	Flags,6	; only a small adjustment needed
	    
; These used only by TMR2 interrupt
;
Dithr       RES     2   ; Dither control working value
PWMtemp     RES     1   ; used to line up 2 bits for the PWM
;
; PWM is the value used for setting the OCXO control voltage.
; 0x000000 is 0 Volt, 0xFA0000 is 5 Volt. See TMR2 interrupt for details.
;
PWM         RES     3   ; PWM - 3 byte value for pulse duty cycle
PWMcalc	    RES	    3	; Calculated before temperature compensation
PWMwork	    RES	    3	; after compensation before loading to PWM
;
; Charge variables
;
ChTime	    RES	    2	; LoopC value to start charge or discharge
; ChgLen is smoothed total of 8 TMR1 values. TMR1 counts in 3.2uS intervals.
; ChgLen+1,2 is	therefore 32*average TMR1 value - counts in 102.4uS intervals
; This is close enough to 100uS to set the precharge time
ChgLen      RES     3
ChgAvg	    RES	    2	; smoothed TMR1 value
ChgWork	    RES	    1	; Work area to do smoothing
DchTime	    RES     2   ; LoopC value to start discharge
; DisVal varies between 0 and 255, depending on when the GPS 1pps arrives
; the target is 128.5 (fluctuating between 128 and 129)
DisVal      RES     1   ; discharge variable
PreInc      RES     1   ; a change applied to DisVal each second
DisOff      EQU     2   ; offset 200uS added to DisVal to get start of discharge
ChgOff	    EQU	    50	; 5mS, target gap between charge and discharge
;
; LED start/end clocks tested in FlagWait
;
Led1on      RES     1   ; LoopC+1 count to turn LED1 on
Led1of      RES     1   ; " " " LED1 off
Led2on      RES     1   ; " " " LED2 on
Led2of      RES     1   ; " " " LED2 off
;
Mloop       RES     1   ; Mainline loop variable
Seconds     RES     1   ; delay in seconds
;
; data collection for a measurement period
;
Samples     RES     3   ; number of samples to take in the period
Accum       RES     3   ; sum of all DisVal for the period
Counted     RES     3	; samples used for linear regression
SumXY       RES     4   ; sum of DisVal*subperiod(= 1 to 42)
;
; RX interrupt variables
;
RXOff	    RES	    1 ; offset for computed GOTO
RXcount	    RES	    1 ; (1) count commas (2) count checksum characters
RXChkSum    RES	    1 ; checksum of GPS message 
; -------------- bank 1 --------------
;
bnk1B      udata   0xC0 ; leave room for Maths data
;
; Variables used to zero in on correct lock voltage
; note order of Vxx and Cxx should be retained as they are
; sometimes manipulated as a block using indirect addressing
;
Vslow       RES     3
Cslow       RES     3
Vfast       RES     3
Cfast       RES     3
;
; Slope is the relationship between PWM and frequency.
; During setup, it is the PWM change needed to change the
; cycles (frequency) of the oscillator by 4 cycles in 84 seconds.
; After the delay circuit is calibrated, it is recalculated
; as the PWM change to require a delay change of one delay
; unit (approximately 20uS) in 84 seconds. In calculations
; it is assumed to be a signed number to accomodate +ve
; or -ve slopes.
Slope       RES     4
;
; DeltaP is the calculated change in delay units over the
; measurement period. It is stored as 2byte integer:2byte fraction
;
DeltaP      RES     4
;--------- end data definitions ---------
      
	    code

; #### program starts here ####

; program starts here on power up or reset
	ORG 0x000
	GOTO MAIN

;--------- Interrupts start here --------
; all interrupts redirect the program to here
	ORG 0x004
;
; save state of whatever was interrupted (As per Data Sheet)
;
	MOVWF   SaveW       ; save W in common memory
	SWAPF   STATUS,W    ; get state without disturbing it
	CLRF    STATUS      ; clears bank flags (change to bank 0)
	MOVWF   SaveS       ; save the (swapped) state
	;
	BTFSS   PIR1,TMR2IF ; skip if TMR2 interrupt
	GOTO    INTrb0	    ; not TMR2, go to next test
;
; ----------- start TMR2 interrupt process ---------
;
; This occurs every 100uS.
; On every entry:
;   1. Dither the PWM duty cycle
;   2. Update the count of TMR2 interrupts
;   3. Checks if it's time to start charging/discharging  
; After 10,000 interrupts (one second), the count is reset.
; Note that during set up, the RB0 interrupt can reset the TMR2
; count so the period is not always 1 second. Once an oscillator/GPS
; lock is established, the period is always 10,000 interrupts at 100uS
; intervals (= 1000 CPU clock cycles, = 250 instruction times
;
	BCF     PIR1,TMR2IF ; clear the interrupt flag

; Timer 2 determines the length of each PWM output. The duty cycle of
; each pulse is dithered each interrupt. In software, the PWM value is
; held as 24 bits. The most significant 10 bits are loaded into the PWM
; duty cycle register. The least significant 14 bits of the 3 byte PWM
; value are repeatedly added to the value Dithr. If Dithr overflows to
; the 15th bit then the duty cycle is increased by 1 bit for the next
; PWM pulse.

	MOVF    PWM,W   ; Add the least significant
	ADDWF   Dithr,F   ; 14 bits of PWM to Dithr
	MOVF    PWM+1,W
	ANDLW   0x3F ; truncate to top 6 of 14 bits
	BTFSC   STATUS,C
	ADDLW   0x01 ; add in carry
	ADDWF   Dithr+1,F ; add to the current Dithr
	MOVF    PWM+1,W ; now work out the top 10 bits
	ANDLW   0xC0 ; get least significant two bits of 10
	BTFSC   Dithr+1,6 ; was there 14 bit overflow
	ADDLW   0x40 ; yes, increase duty cycle 1 bit
	BCF     Dithr+1,6 ; and clear the bit from the Dithr value
	MOVWF   PWMtemp ; save 2 bits for inserting in CCP1CON
	MOVF    PWM+2,W ; top 8 bits of duty cycle
	BTFSC   STATUS,C
	INCF    PWM+2,W ; add 1 if there was a carry from bottom 2 bits
	MOVWF   CCPR1L ; and store it as top 8 bits of next pulse width
	CLRC           ; move PWMtemp bits <6:7> down to <4:5>
	RRF     PWMtemp,F ; line up to insert in CCP1CON
	RRF     PWMtemp,F
	MOVF    CCP1CON,W
	ANDLW   0xCF ; turn off previous <4:5>
	IORWF   PWMtemp,W ; insert new <4:5>
	MOVWF   CCP1CON ; stores bottom 2 bits of next pulse width
;
; update the loop counters
;
	INCFSZ  LoopC,F
	GOTO    Int2cap
	INCFSZ  LoopC+1,F
	GOTO    Int2cap
;
; end of 1 second - set the 1pps output
;
; This happens a fixed time after the TMR2 interrupt, so the 1pps output
; should not jitter.
;
	BSF     MakePPS ; trigger the monostable to make a 1pps
; set flags for mainline program
	BSF     LoopEnd ; end of a second
;
; start another second - 40 outer loops, the first of 16 TMR2
; interrupts and the rest of 256 = 256 * 39 + 16 = 10,000 loops.
; For programming considerations, the counter counts up to zero
;
	MOVLW   0xD8 ; d'-40'
	MOVWF   LoopC+1
	MOVLW   0xF0 ; d'-16'
	MOVWF   LoopC
	CLRF	Led1on ; clear here, so mainline has to
	CLRF	Led2on ; set up LEDs each second
	CLRF	ChTime+1 ; Do this so charge/discharge has to be set
			 ; up every time
	BCF     MakePPS ; end 1pps - monostable should have fired
	GOTO    TXtest ; see if there is diagnostic data to output

; Check if it is time to charge/discharging the detector capacitor

Int2cap:
	MOVF    ChTime,W  ; compare LSB first, least likely to match
	SUBWF   LoopC,W
	BTFSS   STATUS,Z
	GOTO	EndInt
	MOVF    ChTime+1,W    ; then the MSB
	SUBWF   LoopC+1,W
	BTFSS   STATUS,Z
	GOTO	EndInt
; time to charge/discharge. Distinguish by TRISA,DisCharge - if
; this is output (0) then start charge. If it is input (1), it
; was set this way by RB0 interrupt so start discharge
; both DisCharge and PreCharge should be high impedence after
; charging the capacitor. Always discharge through DisCharge.

	errorlevel -302 ; Disable banking message
	BSF     STATUS,RP0  ; to bank 1
	BTFSS	TRISA,DisCharge ; discharging?
	GOTO	Int2Chg ; yes, switch to charging
; no, initiate discharge
	BCF     TRISA,DisCharge ; start discharging
	errorlevel +302 ; Enable banking message

	BCF     STATUS,RP0  ; to bank 0
	BCF     PORTA,DisCharge ; 0v to discharge
; should drop the detector below 4v so can enable interrupt
	GOTO	EnabRB0
; no need to switch banks, status restored before exit
Int2Chg:
; discharging through DisCharge. Set it high impedence, start charge

	errorlevel -302 ; Disable banking message
	BSF     TRISA,DisCharge ; set as not output
	BCF     TRISA,PreCharge   ; set as output
; and start the timer
	errorlevel +302 ; Enable banking message

	BCF     STATUS,RP0  ; to bank 0
	BSF	PORTA,PreCharge ; make sure we are charging
	CLRF    TMR1H   ; set the timer to zero
	CLRF    TMR1L
	BCF	PIR1,TMR1IF
	BSF     T1CON,TMR1ON ; start timer 1 counting
; set the time to start discharge
	MOVF	DchTime,W
	MOVWF	ChTime
	MOVF	DchTime+1,W
	MOVWF	ChTime+1
EnabRB0:
    	BCF	INTCON,INTF ; clear interrupt (if any)
	BSF	INTCON,INTE ; and allow the next one
;
EndInt:
;
; we are done here
;
	SWAPF   SaveS,W ; load previous state in W (unswapping on load)
	MOVWF   STATUS  ; restore STATUS register (maybe changing bank)
	SWAPF   SaveW,F ; swap nibbles preparing to reload W
	SWAPF   SaveW,W ; swap back again into W
	RETFIE
;
; ------------ end TMR2 interrupt process ----------
;
INTrx:
    BTFSS   PIR1,RCIF	; skip if Asynchronous Receive interrupt
    GOTO    INTtx
;
; ----------- start RX interrupt process ---------
;
; Although this interrupt is later in the interrupt chain
; it is positioned lower in memory as it has a computed GOTO,
; preferred in first 256 instructions
;
; Read and process received data - example
;   $GPRMC,123519,A,4807.038,N,01131.000,E,022.4,084.4,230394,003.1,W*6A
;
; Looking for A in the third field indicating valid data received
RXRead:
    BTFSC   RCSTA,OERR ; check for an overrun error
    GOTO    RXOver
    MOVF    RXOff,W ;load offset in w reg
    ADDWF   PCL,F ; add offset to pc to generate a computed goto
RXtabS:
; This series of instructions is processed starting with the one
; at RXtabE, which transforms the offset in W to the character $
; which is then compared with the received character. On a match,
; the offset is decreased, W is transformed to a pattern which
; when transformed by the next instruction yields the character G.
; This process continues (assuming all characters match) until the
; header $GPRMC is identified. The remaining targets of the computed
; GOTO are GOTOs that process the rest of the message.
    GOTO    RXCheck ; compare checksum with calculated
    GOTO    RXProc ; process the body of the message
    GOTO    RXHddrOK ; after a valid header was read
; this code identifies the header $GPRMC. W is converted by
; one or more XOR to the expected character
    XORLW   A'C' ^ ($ - RXtabS) ^ ($ - RXtabS + 1) ^ A'M'
    XORLW   A'M' ^ ($ - RXtabS) ^ ($ - RXtabS + 1) ^ A'R'
    XORLW   A'R' ^ ($ - RXtabS) ^ ($ - RXtabS + 1) ^ A'P'
    XORLW   A'P' ^ ($ - RXtabS) ^ ($ - RXtabS + 1) ^ A'G'
    XORLW   A'G' ^ ($ - RXtabS) ^ ($ - RXtabS + 1) ^ A'$'
RXtabE:
    XORLW   A'$' ^ ($ - RXtabS)
    DECF    RXOff,F	; assume it's the right character
    XORWF   RCREG,W ; get the character
    BTFSC   STATUS,Z ; was it the character we want?
    GOTO    EndInt  ; yes
; Reset the header offset and exit. This is used whenever
; a non conforming condition is detected.
Hreset:
    MOVLW   RXtabE - RXtabS
    MOVWF   RXOff
    GOTO    EndInt

; processing the body of the message

RXHddrOK:
; here for the first character of the body, do some housework
    BCF	    ValidGPS ; will be set if there's an A in the right place
    MOVLW   -3 ; set so that comma count is tested by INCFSZ
    MOVWF   RXcount ; count commas
; set the checksum with the value of characters processed so far
    MOVLW   A'G' ^ A'P' ^ 'R' ^ 'M' ^ 'C'
    MOVWF   RXChkSum
    DECF    RXOff,F	; points to body processing
RXProc:
    MOVF    RCREG,W ; get the character
    XORLW   A'*'    ; check if it's an asterisk
    BTFSC   STATUS,Z
    GOTO    RXaster ; asterisk is the last character of the body
    XORLW   A'*'    ; reinstate original character
    XORWF   RXChkSum,F	; and update the checksum
    XORLW   A','	; was it a comma
    BTFSC   STATUS,Z
    INCF    RXcount,F ; was a comma, increase comma count
; if it was a comma, no need to exit, the following does it anyway
    XORLW   A',' ^ A'A' ; was it an A
    BTFSC   STATUS,Z ; not an A
    INCFSZ  RXcount,W ; is an A, is it after the second comma
    GOTO    EndInt ; only interested in an A after 2nd comma
    BSF	    ValidGPS ; an A in the status field - GPS OK
    GOTO    EndInt ; only interested in an A
RXaster:
    DECF    RXOff,F ; asterisk, body finished, redirect to checksum
    MOVLW   0x03
    MOVWF   RXcount ; 2 checksum chars and a CR
    GOTO    EndInt

RXCheck:
; Process three characters. The first two are assumed to be the
; checksum, the next should be a carriage return. 
    MOVF    RCREG,W ; get the character
    DECFSZ  RXcount,F   ; have we processed two previous characters?
    GOTO    RXdoSum ; no, process a checksum character
    XORLW   0x0D    ; Carriage Return?
    BTFSC   STATUS,Z
    MOVF    RXChkSum,F
    BTFSS   STATUS,Z ; skip if checksum OK
    GOTO    Hreset
; Received a valid $GPRMC message. If the message had an A (active)
; then ValidGPS was set when it was encountered. If there was no A
; but the message is valid assume the GPS has lost signal. Enable or
; disable reception of 1ppS pulses
    BTFSC   ValidGPS
    BSF     ClrPPSin ; enable GPS monostable
    BTFSS   ValidGPS
    BCF     ClrPPSin ; disable GPS monostable
;
    GOTO    Hreset  ; and set up to process more data

RXdoSum:
; convert ascii hex to binary and remove it from the checksum.
    ADDLW   0xBF    ; 'A'->'F' to 0x00 -> 0x05 with overflow
    BTFSS   STATUS,C
    ADDLW   0x07    ; '0'->9 to 0xF6 -> 0xFF
    ADDLW   0x0A    ; 0xF6 ->0x05 to 0x00 -> 0x0F
    SWAPF   RXChkSum,F
    XORWF   RXChkSum,F
    GOTO    EndInt

RXOver:
    BCF	    RCSTA,CREN ; this is what the doco says to do
    BSF	    RCSTA,CREN
    GOTO    EndInt
    
; ----------- end RX interrupt process ---------

INTrb0:
	BTFSS   INTCON, INTF	; skip if RB0 detected
	GOTO    INTrx		; not RB0, go to next test
; RB0 can fire multiple times, only interested in the first then the
; interrupt is disabled but the flag may get set again.
	BCF     INTCON, INTF	; clear the flag
	BTFSS	INTCON,INTE	; was the interrupt enabled?
	GOTO    INTrx		; interrupt not for here
;
; ----------- start RB0 interrupt process ---------
;
; This is triggered by the pulse delay circuit. It can either be:
;    1. When the delay circuit capacitor reaches 4 volts
; or 2. A GPS 1 ppS pulse arriving
	BCF	INTCON,INTE ; first disable the interrupt
; Was the detector capacitor charging, now at 4v? If so, TMR1 will
; be running
	BTFSC   T1CON,TMR1ON  ; skip if not - must be GPS
	GOTO    RB0chg
; GPS pulse - capture the counters
	MOVF    TMR2,W  ; the timer
	MOVWF   Ticks
	MOVF    LoopC,W   ; and the loop counts
	MOVWF   Ticks+1
	MOVF    LoopC+1,W
	MOVWF   Ticks+2
	BSF	GPStick
; Optionally, TMR2 is set to a known state before exiting. TMR2 is set
; to a value that should put the next 1ppS interrupt between TMR2
; interrupts. The TMR2 interrupt happens when TMR2 is zero, and
; inhibits the RB0 interrupt for the time of TMR2 interrupt processing,
; about 50 instructions. A TMR2 value of 150 at the expected time of
; interrupt gives a leeway of approximately 100 instruction times
; either way. By the time the RB0 interrupt processing gets to reading
; TMR2, approximately 16 instruction times have passed, so the desired
; value when read is 166. There are a few more instructions between
; reading and overwriting so the value written is 11 more, or 177.
	BTFSS	ResetCk ; if requested
	GOTO	EndInt
	MOVLW   TargVal+0x0B ; timer reset to 177
	MOVWF   TMR2
	MOVLW   RB0Lo	; LoopC set so the counters hit
	MOVWF	LoopC	; zero in three TMR2 interrupts
	MOVLW   RB0Hi
	MOVWF   LoopC+1
	GOTO	EndInt

; RB0 triggered by PreCharge reaching 4V. Stop the charging, stop TMR1
; so capturing the time to charge.

RB0chg:
	BCF     T1CON,TMR1ON ; stop timer 1 counting
	BSF     STATUS,RP0  ; go to Bank 1
; could be charging through PreCharge or Varist. Set both high impedence
	   errorlevel -302 ; Enable banking message

	BSF     TRISA,PreCharge   ; set as not output
	BSF     TRISA,DisCharge ; set as not output
   errorlevel +302 ; Enable banking message
; EndInt cleans up the bank setting
	GOTO	EndInt

; ------------ end RB0 interrupt process ---------
;
; An unexpected interrupt. Should not happen. Since the source
; is unknown this might not work. Try to flash error 12 on
; the LEDs

INTunknown:
    MOVLW   0xC0 ; = 12 - flash LED2,LED2,LED1,LED1
    MOVWF   ErrVal
    GOTO    ErrEnd

; ######### MAINLINE STARTS HERE #########

MAIN:
	MOVLW   0x07    ; Turn comparators off and
	MOVWF   CMCON   ; enable pins for I/O functions
	MOVLW   0x20    ; clear all of bank 0 and common
	MOVWF   FSR     ; so variables are in a known state
	CLRF    INDF
	INCF    FSR,F
	BTFSS   FSR,7
	GOTO    $-3
	BCF     ClrPPSou ; inhibit the 1pps output
	BCF     ClrPPSin ; and input monostables
	BCF     PORTA,DisCharge ; start with discharged capacitor
;--- BANK 1 Presets ---

	BSF     STATUS, RP0 ; Select Bank1

    errorlevel -302 ; Inhibit banking error messages

; Change Power On Reset default states to what we want
	BCF     OPTION_REG,T0CS ; unassign TMR0 from RA4

; TRISA assignments
;
; RA0 - output port for LED 1
; RA1 - output port for LED 2
; RA2 - Normally discharge capacitor | alternate between tristate (input)
; RA3 - Normally charge capacitor    | and output in program
; RA4 - unused    
; RA5 - reset - not referenced by program - input
; RA6 - output port to make our 1pps
; RA7 - Clock input - not referenced by program
; 

	MOVLW   0xB8        ; Value used to initialize PORTA data direction
	MOVWF   TRISA       ; Set RA0-3,6 as output

; TRISB assignments
; RB0 - Detector input - causes interrupt
; RB1 - Serial in     ( Doco says leave both as inputs
; RB2 - Serial out    ( hardware enables TX as output
; RB3 - PWM output, set as output
; RB4 - output port to disable 1pps output - set as output
; RB5 - output port to disable 1pps input from GPS unit - "
; RB6 - reserved for In Circuit Programming
; RB7 - reserved for In Circuit Programming

	MOVLW   0xC7        ; Value used to initialize PORTB data direction
	MOVWF   TRISB       ; Set RB3-5 as output

; set up the PWM period of 250 instructions
; Once PWM is started, the period is not changed, just the duty cycle

	MOVLW   0xF9 ; d'249' - TMR2 reset every 250 instructions,
	MOVWF   PR2         ; = 1000 Osc cycles = 10KHz
	BSF     PIE1,TMR2IE ; enable TMR2 interrupt

; set up the UART for transmitting

	MOVLW   0x81    ; 129 = 4800 baud
	MOVWF   SPBRG
	MOVLW   0x24
; x 0 1 0 x 1 x x
;   | | |   +- BRGH - use high speed clock
;   | | +----- SYNC - select asynchronous
;   | +------- TXEN - enable transmission
;   +--------- TX9  - select 8-bit transmission
	MOVWF   TXSTA

	BSF	    PIE1,RCIE ; allow receive interrupts

	CALL    TXinit ; initial conditions for transmit buffer

	errorlevel +302 ; Allow banking error messages

	BCF     STATUS, RP0 ; Select Bank0
;--- BANK 0 Presets
	BSF     RCSTA,SPEN  ; Serial Port Enable bit
;
; RXinit: Set the header offset to the correct value.
;         Must be called before interrupts are enabled.
;
	MOVLW   RXtabE - RXtabS ; header offset
	MOVWF   RXOff
; Set up Timer 1
	MOVLW   0x30
	MOVWF   T1CON
; x x 1 1 0 x 0 0
;     -+- |   | +- TMR1ON - not started until precharging
;      |  |   +--- TMR1CS - Internal clock (FOSC/4)
;      |  +------- T1OSCEN - Oscillator is shut off (not used)
;      +---------- 11 = 1:8 Prescale value

; set up TMR2 and PWM. 
	MOVLW   0x0C        ; Value for PWM mode
	MOVWF   CCP1CON
; Set up the duty cycle. The variable PWM overwrites CCP registers
; every TMR2 interrupt. PWMcalc updates PWM each second
	MOVLW   0x7D ; d'125' - PWM Duty Cycle to start, 50%
	MOVWF   PWM+2
	MOVWF   PWMcalc+2
	CLRF    TMR2
; need a value in the timer loop or there will be a long delay
; before it reaches zero
	MOVLW   0xD8 ; d'-42' - should reach loop end about 1 second,
	MOVWF   LoopC+1 ; plenty of setup time
; start timer 2 here.  Should see a 10KHz PWM signal on RB4
	BSF     T2CON,TMR2ON ; start timer
	BCF     PORTA,DisCharge ; no charge to start
	BSF     LED1 ; let user know we got this far
	MOVLW   0xC0
; 1 1 0 0 0 0 0 0
; | |   +- INTE - allow RB0 interrupt - not yet
; | +----- PEIE - allows peripheral interrupts (specifically TMR2)
; +------- GIE  - allow any enabled interrupt
	MOVWF   INTCON

;=====================================================================
; Test the detector capacitor. If the hardware is OK, this test is
; optional.
; Apply 5v to the detector network which will eventually fire RB0
; interrupt at 4v. When used as a detector the capacitor is discharged
; from 4V starting from 1 to 256 TMR2 interrupts before the expected
; arrival of a GPS 1pps pulse with a target around 128 (or 12.8mS).
; A residual of around 2V after 12.8mS is equivalent to a charge from
; 0 to 4V in 30mS so look for a value larger than that.
; 30mS to 45mS is accepted to allow for component tolerance.
;=====================================================================

	CALL	Stage ; = 1 - testing the delay RC
; The capacitor is charged/discharged 12 times, the first four are
; ignored and time to charge is accumulated for the next 8 seconds.
; Mtemp was zeroed when Bank 0 was cleared so no need to do it here.
; RA2 (DisCharge pin) was set to output in setup.
	MOVLW   d'12'
	MOVWF   Mloop
	BCF	LoopEnd
CalDis1:
	BTFSS   LoopEnd ; spin waiting for the end of a second
	GOTO    CalDis1
	BCF	LoopEnd
; Start charging
	CLRF    TMR1H   ; set the timer to zero
	CLRF    TMR1L
	BCF	PIR1,TMR1IF
	BSF     T1CON,TMR1ON ; start timer 1 counting
	BSF     PORTA,DisCharge ; +5v on discharge pin
	BCF	INTCON,INTF ; clear interrupt (if any)
	BSF	INTCON,INTE ; and allow next
CalDis2:
	BTFSC	PIR1,TMR1IF ; test if TMR1 has overflowed (>200mS)
	CALL    ErrEnd ; did not interrupt in time
	BTFSC	T1CON,TMR1ON ; has RB0 interrupt stopped the timer
	GOTO    CalDis2 ; spin waiting
; RB0 also sets ports as input. Change this and start discharge
	BCF     PORTA,DisCharge
	BSF     STATUS,RP0  ; to bank 1
		errorlevel -302
	BCF     TRISA,DisCharge ; start discharging
		errorlevel +302
	BCF     STATUS,RP0  ; to bank 0
;	MOVLW	TMR1L
;	MOVWF	FSR
;	MOVLW	0x02 | (('c' & 0x1F)<<3)
;	CALL	TXhex
; test if count =< 8, skip if it isn't
	MOVF	Mloop,W
	SUBLW	d'8'
	BTFSS	STATUS,C ; skip if OK
	GOTO	CalDis3
; accumulate TMR1
	MOVF    TMR1L,W
	ADDWF   Mtemp,F
	MOVF    TMR1H,W
	BTFSC   STATUS,C
	INCFSZ  TMR1H,W
	ADDWF   Mtemp+1,F
	BTFSC   STATUS,C
	INCF    Mtemp+2,F
CalDis3:
	DECFSZ  Mloop, F
	GOTO	CalDis1
; TMR1 now accumulated 8 times
; TMR1 clocks at 312.5KHz so should have between
; 312.5 * 8 * 30 (75000) and 312.5 * 8 * 45 (112500) clocks
; test this by dividing Mtemp by 2 and testing the middle byte
; between 147 (30.1mS) and 220 (45mS), ignoring LSB
	BCF	STATUS,C
	RRF	Mtemp+2,W
	IORLW	0x00 ; should be zero, test it
	BTFSS   STATUS,Z
	CALL    ErrEnd
	RRF	Mtemp+1,W
; test the value - from 30mS to 45mS
	ADDLW   0x23 ; d'35' - carries if more than 220 (>45mS)
	BTFSC   STATUS,C
	CALL    ErrEnd
	ADDLW   0x4A ; d'54' - carries if originally more than 147 (30mS)
	BTFSS   STATUS,C
	CALL    ErrEnd
; diagnostic
	MOVLW	Mtemp
	MOVWF	FSR
	MOVLW	0x03 | (('c' & 0x1F)<<3)
	CALL	TXhex
;=====================================================================
; Test the phase detector using the PreCharge resistor. If the hardware
; is OK, this test is optional. A value is needed in ChgAvg for subsequent
; use. It can be determined by this routine, or set a fixed value based on
; the expected highest value of the charging resistor. It is a thermistor
; and the charge time can vary according to temperature. ChgAvg is
; recalculated during running so the start value is not critical.
; Charging via PreCharge, Discharging via DisCharge. 	
;=====================================================================
	CALL	Stage ; = 2 - testing thermistor
	BSF     PORTA,PreCharge ; ready to charge, 5v when output
; DisCharge 0v and enabled at this point. Leave at 0v to discharge,
; enable (output) and disable (input) as needed
	CLRF	Mtemp
	CLRF	Mtemp+1
	CLRF	Mtemp+2
; As before the capacitor is charged/discharged 12 times, the first four are
; ignored and time to charge is accumulated for the next 8 seconds.
	MOVLW   d'12'
	MOVWF   Mloop
	BCF	LoopEnd
CalChg1:
	BTFSS   LoopEnd ; spin waiting for the end of a second
	GOTO    CalChg1
	BCF	LoopEnd
; Start charging
	CLRF    TMR1H   ; set the timer to zero
	CLRF    TMR1L
	BCF	PIR1,TMR1IF
	BSF     T1CON,TMR1ON ; start timer 1 counting
	BSF     STATUS,RP0  ; to bank 1
		errorlevel -302
	BSF     TRISA,DisCharge ; stop any discharging
	BCF     TRISA,PreCharge ; start charging
		errorlevel +302
	BCF     STATUS,RP0  ; to bank 0
	BSF     PORTA,PreCharge ; +5v on discharge pin
	BCF	INTCON,INTF ; clear interrupt (if any)
	BSF	INTCON,INTE ; and allow the next one
CalChg2:
	BTFSC	PIR1,TMR1IF ; test if TMR1 has overflowed (>200mS)
	CALL    ErrEnd ; did not interrupt in time
	BTFSC	T1CON,TMR1ON ; has RB0 interrupt stopped the timer
	GOTO    CalChg2 ; spin waiting
; RB0 sets ports as input. Change this to start discharge
	BSF     STATUS,RP0  ; to bank 1
		errorlevel -302
	BCF     TRISA,DisCharge ; start discharging
		errorlevel +302
	BCF     STATUS,RP0  ; to bank 0
;	MOVLW	TMR1L
;	MOVWF	FSR
;	MOVLW	0x02 | (('d' & 0x1F)<<3)
;	CALL	TXhex
; test if count =< 8, skip if it isn't
	MOVF	Mloop,W
	SUBLW	d'8'
	BTFSS	STATUS,C ; skip if OK
	GOTO	CalChg3
; accumulate TMR1
	MOVF    TMR1L,W
	ADDWF   ChgLen,F
	MOVF    TMR1H,W
	BTFSC   STATUS,C
	INCFSZ  TMR1H,W
	ADDWF   ChgLen+1,F
	BTFSC   STATUS,C
	INCF    ChgLen+2,F
CalChg3:
	DECFSZ  Mloop, F
	GOTO	CalChg1
; Need a ChgAvg for FlagWait temperature compensation even if TMR1
; not used
; don't worry about C, it can't make it into ChgAvg
	RRF	ChgLen+2,W ; three shifts to divide by 8
	MOVWF	ChgWork
	RRF	ChgLen+1,W
	MOVWF	ChgAvg+1
	RRF	ChgLen,W
	MOVWF	ChgAvg
	BCF	STATUS,C
	RRF	ChgWork,F
	RRF	ChgAvg+1,F
	RRF	ChgAvg,F
	BCF	STATUS,C
	RRF	ChgWork,F
	RRF	ChgAvg+1,F
	RRF	ChgAvg,F
; TMR1 now accumulated 8 times
; As an approximation, divide by 256 (2.4% error) to get average 100uS
; intervals to charge. Can just take Mtemp+1, Mtemp+2
; diagnostic
	MOVLW	ChgLen+1
	MOVWF	FSR
	MOVLW	0x02 | (('d' & 0x1F)<<3)
	CALL	TXhex
;=====================================================================
; For the next tests, the detector precharge is set to approx 2v using
; the voltage reference. Enable receiving GPS pulses, wait for GPS.
; From here on, basic end of second (and beginning of next) processing
; is always by the subroutine FlagWait. 
;=====================================================================
	BCF	LoopEnd ; wait for cap to discharge
VRwait:
	BTFSS   LoopEnd ; spin waiting for the end of a second
	GOTO    VRwait
	BCF	LoopEnd
	BSF     STATUS,RP0  ; go to Bank 1

	errorlevel -302 ; Inhibit banking error message

	BSF     TRISA,RA2   ; sets RA2 as not output
	MOVLW   0xEA        ; enable VREF
; 1 1 1 0 1010
; | | | |  +-- set Vref to (Vdd * 10)/24 = approx 2.1v
; | | | +-- unused
; | | +-- low range
; | +-- Vref output on RA2
; +-- Vref circuit powered on
	MOVWF   VRCON
; The 1ppS from the GPS is enabled or blocked by ClrPPSin. At startup
; it is blocked. GPS messages are received on the RX input and are
; decoded if interrupts are enabled. ClrPPSin is enabled if a decoded
; message indicates the GPS is receiving valid input.
	BSF	PIE1,RCIE ; enable receive interrupts

	errorlevel +302 ; Allow banking error message

	BCF     STATUS,RP0 ; back to Bank 0
; able to receive GPS signal from here on
	BSF	VRon ; FlagWait uses this
	CALL	Stage ; = 3 - waiting for GPS
	BSF	ResetCk ; request counters reset each RB0 interrupt
	BSF     RCSTA,CREN  ; Start the UART receiver
   	CLRF    TstLoop
; Flash LED2 on 1/2 second off 1/2 second while waiting for GPS
; set the off time, set on in WaitGPS if no GPS
	MOVLW   0xF6 ; d'-10'
	MOVWF   Led2of
; if there is GPS, FlagWait sets LED1 on, set the off time
	MOVLW   0xED
	MOVWF   Led1of  ; d'-19'
	MOVLW   0x02 ; delay 2 seconds to let LoopC sync with GPS
	MOVWF   Seconds
NoGPS:
	MOVLW   0xE2 ; d'-30'
	MOVWF   Led2on
WaitGPS:
; Wait for the GPS signal
	CALL    FlagWait ; fall through at end of second
	BTFSS   GPStick
	GOTO	NoGPS
; The Loop values should be set up correctly by the first GPS pulse
; allow 2 loops
	DECFSZ  Seconds,F
	GOTO    WaitGPS
;=====================================================================
; Now receiving 1pps GPS signal. This section gives a visual indication of
; the oscillator/1ppS synchronisation with the control voltage set to approx
; 2.5V. If the oscillator is exactly 10MHz, the LEDS flash in unison.
; If not, LED2 flashes before or after LED1, the difference between flashes
; an indication of the frequency error.
; There is usually a difference until the oscillator has warmed up. It may
; be necessary to wait a while and restart the processor after a few minutes.
; This section is optional, it is not required for calibration.
;=====================================================================
	MOVLW   0x3C ; loop through 60 seconds (could be changed)
	MOVWF   Seconds ; work loop
	CALL	Stage ; = 4 - GPS indicator phase
IdleMiss:
	CLRF    Led2on ; no LED2 if GPS miss
IdleTest:
	CALL    FlagWait
	BTFSS   GPStick
	GOTO    IdleMiss ; missed GPS pulse
; decide what flash to put on Led2. If the frequency is drifting
; then flash it earlier or later than Led1.
; Use Mtemp as several one byte work areas
	CLRF    Led2of ; delay between Led1 and Led2
	MOVLW   0x01
	MOVWF   Mtemp ; start with first two Fibonacci numbers
	MOVWF   Mtemp+1
	MOVF	Tcomp,W
	BTFSC	STATUS,Z
	GOTO	SkipFib ; difference zero, flash together
; get abs(Tcomp)
	BTFSC   Tcomp,7 ; test the sign bit
	SUBLW   0x00 ; make it positive if it was -ve
	MOVWF   Mtemp+2
	INCF    Led2of,F ; minimum count = 2
Ledshift:
; Use the Fibonacci series to set the advance/retard of Led2
; Tcomp   0     1     2    3-4   5-7   8-12 13-20 21-33  etc..
; Shift   0     2     3     4     5     6     7     8
; This is approximately logarithmic
	INCF    Led2of,F
; generate the next number in the series
	MOVF    Mtemp,W   ; Mtemp, Mtemp+1 are successive Fibonacci numbers
	ADDWF   Mtemp+1,W ; add them to get next in series
	XORWF   Mtemp+1,W ; a difference pattern of Mtemp and Mtemp+(Mtemp+1)
	XORWF   Mtemp+1,F ; converts Mtemp+1 to Mtemp and Mtemp+(Mtemp+1)
	XORWF   Mtemp+1,W ; converts W to the previous value of Mtemp+1
	MOVWF	Mtemp
	SUBWF   Mtemp+2,W ; compare difference with previous Fibonacci
	BTFSC   STATUS,C  ; result -ve, end of test
	GOTO    Ledshift  ; go again
	MOVF    Led2of,W
	BTFSC   Tcomp,7   ; make the adjustment the same sign
	SUBLW   0x00      ; as Tcomp
	MOVWF   Led2of
; make Led2 an offset of Led1
SkipFib:
	MOVF    Led1on,W
	ADDWF   Led2of,W
	MOVWF   Led2on
	MOVF    Led1of,W
	ADDWF   Led2of,F
;
	DECFSZ  Seconds, F
	GOTO    IdleTest ; go back for more seconds
;=====================================================================
; Now test with a 10% PWM signal (approx 0.5V control voltage) and with
; 90% (approx 4.5V). This tests the pulling limits of the oscillator.
; It should run fast with one and slow with the other (the program does
; not care which way round). This determines if a lock is possible.
; The most likely cause of failure is the oscillator is not yet stable.
;=====================================================================
	CALL	Stage  ; = 5 - testing oscillator pulling limits
	BSF     STATUS,RP0	; bank 1 to set up work areas
	CLRF    Vslow+2     ; if all is well, these bytes will be
	CLRF    Vfast+2     ; filled in with non-zero values
	BCF     STATUS,RP0	; to bank 0
	MOVLW   0x19        ; PWM Duty Cycle for first test, 25 = 10%
	MOVWF   PWMcalc+2
	CALL    FreqErr	    ; get the error
	CALL    SavLims	    ; decide if to store it as fast or slow
	MOVLW   0xE1        ; PWM Duty Cycle for second test, 225 = 90%
	MOVWF   PWMcalc+2
	CALL    FreqErr	    ; get the error
	CALL    SavLims	    ; decide if to store it as fast or slow
; Test to see if one PWM was moved to Vslow, the other to Vfast
	BSF     STATUS,RP0	; bank 1
	errorlevel -302 ; Disable banking message
	MOVF    Vslow+2,F
	BTFSS   STATUS,Z	; Vslow not zero ?
	MOVF    Vfast+2,F		
	errorlevel +302 ; Enable banking message
	BTFSC   STATUS,Z	; Vfast not zero
; If successful, go to where the value of Slope is calculated
; and start zeroing in on the target
	CALL    ErrEnd  	; can't lock within PWM range
;;=====================================================================
; This section looks for control voltages that cause the oscillator
; to be 'pulled' +- approximately 1 part in 10^8 from the lock frequency.
; A deviation of 1 part in 10^8 corresponds to a change of one
; instruction time of 400nS over 40 seconds. In practice 42 seconds is
; used so the value of Slope is compatible with later processing.
; 1) The control voltage/frequency change relationship is calculated.
; 2) The actual frequency changes are tested against the target of a
;    drift of 1 instruction time per 42 seconds. If they are within
;    tolerance control moves to the next section.
; 3) Otherwise the high and low voltages are refined and trialled,
;    Control then passes back to step 1)
; The test is quite sensitive and it is not uncommon to do several trials
; each taking about 8-1/2 minutes before results are satisfactory.
; There is an arbitary limit of 10 trials, after which the section is
; regarded as failed and the program stops with an error
;=====================================================================
	BCF     STATUS,RP0	; bank 0
	MOVLW   0x0A        ; loop a maximum of 10 times to try and
	MOVWF   TstLen     ; converge on usable values.
	CALL	Stage  ; = 6 - convergance test
TryVals:
; compute a new <control volts>/<frequency change> slope
; slope = (Vfast - Vslow)/(Cfast - Cslow) expressed
; as change of 24-bit PWM value to get 400nS drift per 84 seconds
; 
; Note that C was stored *256 but for a period of 42
; seconds, equivalent to *128 for 84 seconds.
; so actual equation is:
; ((Vfast - Vslow)*128)/(Cfast - Cslow)
	BSF     STATUS,RP0	; bank 1 for Arithmetic
	MOVLW   Cfast       ; push Cfast
	CALL    Mpush3
	MOVLW   Cslow       ; push Cslow
	CALL    Mpush3
	CALL    Msubtract    ; Cfast - Cslow
	MOVLW   Mtemp   ; pop for convergance test later
	CALL    Mpop3
	MOVLW   Vfast       ; push Vfast
	CALL    Mpush3U
	MOVLW   Vslow       ; push Vslow
	CALL    Mpush3U
	CALL    Msubtract    ; Vfast - Vslow
	MOVLW   0x07        ; shift by 7 = multiply by 2^7 (128)
	CALL    Mshift    ; (Vfast - Vslow)*128
	MOVLW   Mtemp
	CALL    Mpush3   	  ; push (Cfast - Cslow) back
	CALL    Mdivide      ; get final result (slope)
	MOVLW   Slope   	; result - new value for slope
	CALL    Mpop4
; Test here if we are converging on the values we want.
;
; The aim is to determine PWM values that get 1 instruction time (400nS)
; oscillator drift per 42 seconds either side of an assumed PWM that
; gives a zero frequency error.
; Ideally Cfast should be 1 (*256) and Cslow should be -1 (*256)
; but any non linearity may prevent reaching exact values so instead
; look for Cfast between 0.75 and 1.25 and a Cfast - Cslow around 2
; (all multiplied by 256). This is adequate to proceed to the next step
	MOVF    Cfast+2,F   ; looking for little endian xx yy 00
	BTFSS   STATUS,Z	; msb = 0?
	GOTO    NoConv	; no, fail
	BCF     STATUS,C
	RRF     Cfast+1,W	; if yy 0 or 1, W = 0
	IORLW	0x00        ; test for zero, any test will do
	BTFSS	STATUS,Z
	GOTO	NoConv	; fail, yy was > 1
	MOVF	Cfast,W
	BTFSS	STATUS,C ; skip if yy was 1
	ADDLW	0x40 ; yy = 0, this will make 0.75 to 1 = 0 to 0.25
	ADDLW	0xBF ; will overflow for any value > 0.25
	BTFSC	STATUS,C
	GOTO	NoConv
; now test Cfast - Cslow (held in Mtemp) in the range
; 0x0001E0 to 0x0000220 (i.e. within about 6%)
	MOVF    Mtemp+2,F
	BTFSS   STATUS,Z    ; should be zero
	GOTO    NoConv
	MOVLW   0x20
	ADDWF   Mtemp,W     ; move to range 0x0200 to 0x0240
	BTFSC   STATUS,C
	INCF    Mtemp+1,F
	SUBLW   0x40 ; test 0 to 0x40
	BTFSS   STATUS,C
	GOTO    NoConv ; not in range
	MOVF    Mtemp+1,W
	SUBLW   0x02 ; must be = 2
	BTFSC   STATUS,Z
	GOTO    CalDtr ; converged, go to next test
NoConv:
	BCF     STATUS,RP0	; To Bank 0 for loop test
	DECFSZ  TstLen,F
	GOTO    NewVal   ; not zero, go again
	CALL    ErrEnd ; could not converge in 10 loops?
;
; Calculate zero drift point as Vslow - Cslow * slope. Taking account
; of Cslow is really Cslow*256 and slope is for 84 seconds and the value
; for 42 seconds is needed (i.e. slope * 2) it becomes:
; Vslow - (Cslow * slope)/128
; Then calculate new values of Vfast, Vslow
;
NewVal:
	BSF     STATUS,RP0	; bank 1 for Arithmetic
	MOVLW   Vslow		; push Vslow
	CALL    Mpush3U
	MOVLW   Cslow		; push Cslow
	CALL    Mpush3
	MOVLW   Slope		; push slope
	CALL    Mpush4
	CALL    Multiply	; Cslow * slope
	MOVLW   -7		; shift by -7 = divide by 2^7 (128)
	CALL    Mshift
	CALL    Msubtract	; result - Vslow - (Cslow * slope)/128
	MOVLW   Mtemp		; save calculated voltage for zero error
	CALL    Mpop4
;
; now calculate new values for Vfast, Vslow
; Vfast = Vzero + slope * 2 (*2 converts from 84 to 42 seconds)
; Vslow = Vzero - slope * 2

; Vfast
	MOVLW   Mtemp   ; push Vzero back
	CALL    Mpush4
	MOVLW   Slope   ; push slope
	CALL    Mpush4
	MOVLW	0x01	; now multiply by 2
        CALL    Mshift
	CALL    Madd     ; Vzero + slope should give a new Vhigh
	MOVLW   Vfast
	CALL    Mpop3
; Vslow
	MOVLW   Mtemp   ; push Vzero back
	CALL    Mpush4
	MOVLW   Slope   ; push slope
	CALL    Mpush4
	MOVLW	0x01	; now multiply by 2
        CALL    Mshift
	CALL    Msubtract    ; Vzero - slope should give a new Vslow
	MOVLW   Vslow
	CALL    Mpop3
	BCF     STATUS,RP0	; bank 0 after arithmetic
; and have another test
; slow
	MOVLW   Vslow
	CALL    LoadPWM
	CALL    FreqErr ; returns changes/42 seconds in Mtemp
	MOVLW   Cslow
	MOVWF   FSR
	CALL    SaveCval
; fast
	MOVLW   Vfast
	CALL    LoadPWM
	CALL    FreqErr ; get Ticks value
	MOVLW   Cfast
	MOVWF   FSR
	CALL    SaveCval
	GOTO	TryVals
;=====================================================================
; So far, the detector precharge has been fixed giving a fixed delay
; between a pulse from the GPS and the triggering of RB0. Now the values
; of Vfast and Vslow are close to the target voltage, the detector can
; be calibrated by running two tests of 84 seconds, first with Vfast then
; with Vslow. The difference between the results of the two tests will
; determine the precharge/pulse delay relationship. The precharge
; is measured in LoopC counts between starting discharge and detecting
; the 1ppS. The accepted range is starting discharge 1 LoopC count earlier
; increases the pulse delay range between 14nS to 25nS.
; This is a somewhat arbitary range. It is a compromise between
; sensitivity and ability to track drift.
;=====================================================================
CalDtr:
	BCF     STATUS,RP0	; To Bank 0
	CALL	Stage ;  = 7 - testing detector sensitivity
	BCF	VRon ; used by FlagWait
	BSF	PORTA,PreCharge ; set these in the data latch, no
	BCF     PORTA,DisCharge ; effect until port is output.

    errorlevel -302 ; Turn off banking message

	BSF     STATUS,RP0  ; go to Bank 1
	CLRF    VRCON   ; disable the voltage reference
	BCF     TRISA,DisCharge ; start discharging
	BCF     STATUS,RP0  ; back to Bank 0

    errorlevel +302 ; Turn on banking message

; At the end of the previous test, the oscillator was running fast
; so the first test is done with Vfast (no change)
	CALL    CalWait ; resets loop counter
	CALL    Cal84   ; tracks the change in DisVal for 84 seconds
	SUBLW   0x00    ; the value will be -ve, invert it
; TstLoop+1 used as temporary work area
	MOVWF   TstLoop+1
; switch to slow oscillator
	MOVLW   Vslow
	CALL    LoadPWM
	CALL    CalWait ; drops through when a change is detected
	CALL    Cal84
	ADDWF   TstLoop+1,F ; add and get total Change
; Allow an arbitary minimum value of 64 and maximim 112 which
; equates to a detector sensitivity between 14 and 25 nS
	MOVF    TstLoop+1,W
	ADDLW   0x90 ; = 144 - overflows if > 112 (test less than 14nS)
	BTFSC   STATUS,C
	CALL    ErrEnd
	ADDLW   0x30  ; = 48 - overflows if > 64 (test more than 25nS)
	BTFSS   STATUS,C
	CALL    ErrEnd
; diagnostic
	MOVLW	TstLoop+1 ; output the value via serial port
	MOVWF	FSR
	MOVLW	0x01 | (('j' & 0x1F)<<3)
	CALL	TXhex
; it is now convenient to redefine slope as an adjustment in terms of
; the detector sensitivity - the PWM adjustment to obtain a change
; of one detector count in 84 seconds.
; The calculation is:
; (Slope*4)/TstLoop+1
; The constant 4 is because TstLoop+1 was calculated from a deviation
; of 4 ticks of the LoopC loop
;
; The magnitude of Slope is inversely proportional to the frequency pulling
; range of the oscillator and the sensitivity of the delay circuit.
; 
	BSF     STATUS,RP0	; bank 1 for Arithmetic
	MOVLW   Slope   ; push slope in instruction ticks
	CALL    Mpush4
;	MOVLW	0x02	; now multiply by 4
;	CALL    Mshift
; ** TEST ** the adjustment seems too agressive. So instead of multiply by 4
; multiply by 7/2 (or 3.5)
	MOVLW	0x07
	CALL	MpushLit
	CALL	Multiply
	MOVLW	-1 ; divide by 2
	CALL	Mshift
	MOVF    TstLoop+1,W
	CALL    MpushLit
	CALL    Mdivide ; Slope/TstLoop+1
	MOVLW   Slope
	CALL    Mpop4
; diagnostic
	MOVLW	Slope
	MOVWF	FSR
	MOVLW	0x03 | (('s' & 0x1F)<<3)
	CALL	TXhex

;=====================================================================
; All calibration done. Set an approximate locked PWM, use CalWait to
; reset the detector value and the LoopC values. The system will
; correct any error over a few measurement periods.
;=====================================================================
;
; calculate a target PWM value
	MOVLW   Vslow       ; push Vslow
	CALL    Mpush3U
	MOVLW   Vfast       ; push Vfast
	CALL    Mpush3U
	CALL    Madd
	MOVLW	0xFF ; (-1)
	CALL    Mshift ; divide by 2
	MOVLW   Mtemp
	CALL    Mpop3
	BCF     STATUS,RP0	; bank 0 after arithmetic
	MOVLW   Mtemp   ; target voltage to PWM
	CALL    LoadPWM
; look for next GPS pulse
	CALL	Stage  ; = 8 - running locked
	CALL    CalWait ; reset
; set up Bank0 variables
	MOVLW   0x01    ; determines the initial Test Length (84 seconds)
	MOVWF   PreInc  ; from now on, jog by one every time
	MOVWF   TstLen
	BSF     ClrPPSou  ; allow output of 1ppS now we are locked
;
;=====================================================================
; The program now remains in FinaLoop.
; In the loop, the oscillator is allowed to 'free run' with a fixed
; control voltage regime for a period of time. The period of time is set
; by TstLen. During the period, deviations of the GPS/oscillator difference
; are accumulated. At the end of the period the regime is recalculated
; using the accumulated data. If the errors are small, the calculation
; period may be extended. The longer the period, the smaller the influence
; of GPS jitter on the calculation.
;
; The adjustment is calculated as follows:
; During each period, data is collected to calculate a least squares
; linear regression of the change of differences between the GPS and
; oscillator. Using the average value of the differences and the rate of
; change a difference at end of period can be calculated. A new rate of
; change can then be calculated to reduce the next end of period difference
; to zero. The actual change to PWM is (new rate) - (old rate)
;=====================================================================
FinaLoop:
;--------
;
; The duration of test is calculated from a base of 675 seconds. Initially,
; it is truncated to 675\8 = 84 and may be doubled if error terms are small.
; The duration is int(675*2^(TstLen-4)) seconds
; TstLen     1   2   3   4    5    6    7    8   
; Duration   84 168 357 675 1350 2700 5400 10800
; The program can handle values up to TstLen = 11 (1 day) but the stability
; of the OCXO will determine the practical limit (set by TestLimit)
;
; The base duration is arbitary, and was chosen so longer test periods
; are subdivisions of a day.
;
	BCF     STATUS,RP0
; determine the number of samples (seconds) in this period.
	MOVLW   0x02  ; base of 675 (0x2A3)
	MOVWF   Samples+1
	MOVLW   0xA3  ; base of 675 (0x2A3)
	MOVWF   Samples
	CLRF    Samples+2
	MOVLW   Samples
	BSF     STATUS,RP0	; bank 1 for Arithmetic
	CALL    Mpush3U
	MOVLW   0x04
	SUBWF   TstLen,W
	CALL    Mshift   ; apply shifts
	MOVLW   Samples
	CALL    Mpop3
	BCF     STATUS,RP0  ; end of arithmetic
; Set up to accumulate the results of the detector
	CLRF    TstLoop ; TstLoop counts up - needed by regression
	CLRF    TstLoop+1
	CLRF    TstLoop+2
	CLRF	Accum   ; Accum accumulates detector values
	CLRF	Accum+1 ; to calculate the average error
	CLRF	Accum+2
	CLRF    SumXY   ; SumXY data for linear regression
	CLRF    SumXY+1 ; to calculate the error slope
	CLRF    SumXY+2
	CLRF    SumXY+3
RunAccum:
	CALL    FlagWait
; fall through at end of second, use the data even if no GPS
; pulse was received.

; Derive a number 0 thru 41 from the Loop counter
	BSF     STATUS,RP0	; bank 1 for Arithmetic
	MOVLW   TstLoop   ; get the current count
	CALL	Mpush3U
	MOVF	TstLen,W ; causes right shifts 
	SUBLW	0x00
	CALL	Mshift
	MOVLW	Mtemp
	CALL	Mpop4
	BCF     STATUS,RP0  ; end of arithmetic
;	MOVLW	Mtemp
;	MOVWF	FSR
;	MOVLW	0x01 | (('n' & 0x1F)<<3)
;	CALL	TXhex
	INCFSZ  TstLoop,F ; increment the counter
	GOTO    $+4
	INCFSZ  TstLoop+1,F
	GOTO    $+2
	INCF    TstLoop+2,F
	MOVLW   0x2A ; if = 42, discard these as
	SUBWF   Mtemp,W ; there are not many of them
	BTFSC   STATUS,C ; and they
	GOTO    RAend ; mess up the stats
; move the counter to Counted - samples used
	MOVF	TstLoop,W
	MOVWF	Counted
	MOVF	TstLoop+1,W
	MOVWF	Counted+1
	MOVF	TstLoop+2,W
	MOVWF	Counted+2
; accumulate the value of DisVal
	MOVF    DisVal,W      ; latest DisVal
	ADDWF   Accum,F
	BTFSC   STATUS,C
	INCFSZ  Accum+1,F
	GOTO    $+2
	INCF    Accum+2,F
;	MOVLW	Accum
;	MOVWF	FSR
;	MOVLW	0x03 | (('a' & 0x1F)<<3)
;	CALL	TXhex
	MOVWF   Mtemp+1       ; set DisVal up for multiply
; 8 x 8 multiply of DisVal with a number 1 to 42 derived from TstLoop
; Mtemp = 0 to 41 incremented and put in W
; Mtemp+1 has DisVal which is shifted right, each set bit causing an add
; Mtemp+2, Mtemp+3 are the result
	MOVLW   0x80 ; a flag to stop multiply loop
	MOVWF   Mtemp+2 ; and Mtemp+3 is zero after pop4
	INCF    Mtemp,W ; value 1 to 42 in W
RAmul:
	RRF     Mtemp+1,F ; getting ls bit of DisVal
	BTFSC   STATUS,C
	ADDWF   Mtemp+3,F ; adding W to high byte
	RRF     Mtemp+3,F ; then dividing by 2 (captures any carry)
	RRF     Mtemp+2,F
	BTFSS   STATUS,C ; until the stop bit shuffles out
	GOTO    RAmul
; add this to the X * Y total
	MOVF    Mtemp+2,W
	ADDWF   SumXY,F
	MOVF    Mtemp+3,W
	BTFSC   STATUS,C
	INCFSZ  Mtemp+3,W
	ADDWF   SumXY+1,F
	BTFSC   STATUS,C
	INCFSZ  SumXY+2,F
	GOTO    RAend
	INCF    SumXY+3,F
RAend:
;	MOVLW	SumXY
;	MOVWF	FSR
;	MOVLW	0x03 | (('s' & 0x1F)<<3)
;	CALL	TXhex
	BTFSC   GPStick
; If no GPS signal, skip the DisVal adjustment, reuse the previous
; value of DisVal. Must put some value here because discarding a
; value would change the length of collection and mess up calculations.
	CALL    JogVal  ; ready for next test
; display 
	MOVLW	DisVal
	MOVWF	FSR
	MOVLW	0x01 | (('z' & 0x1F)<<3)
	CALL	TXhex	
; Sanity check - has the oscillator wandered too far?
; this checks DisVal is between 0x40 and 0xBF
	RLF	DisVal,W ; looking at bit 6 and 7
	XORWF	DisVal,W
	ANDLW	0x80
	BTFSS	STATUS,Z ; error if 00 or 11
	GOTO	InRange ; should be 01 or 10
; Out of bounds - force the test length back to minimum
	DECF	TstLen,W ; are we at minimum
	BTFSC	STATUS,Z
 ; already minimum test length, hope it sorts itself out. If not,
 ; JogVal will kill the system when DisVal hits 0
	GOTO	InRange
	MOVLW	0x01
	MOVWF	TstLen
; the system is 0x40 detector units out. Carry still has the sign
; from RLF - C set +ve deviation, C clear -ve deviation
	MOVLW	0x40
	BTFSS	STATUS,C ; +ve deviation needs -ve correction
	SUBLW	0x00
; create a signed mid point (1 byte integer 2 byte fraction)
; for the adjustment routine
	CLRF    Mtemp
	CLRF	Mtemp+1
	MOVWF   Mtemp+2
	MOVLW	Mtemp
	BSF     STATUS,RP0	; bank 1 for Arithmetic
	CALL	Mpush3 ; signed
	GOTO	ForceIt
InRange:
 ; test if enough samples collected	
	MOVF    TstLoop,W
	XORWF   Samples,W
	BTFSS   STATUS,Z
	GOTO    RunAccum
	MOVF    TstLoop+1,W
	XORWF   Samples+1,W
	BTFSS   STATUS,Z
	GOTO    RunAccum
	MOVF    TstLoop+2,W
	XORWF   Samples+2,W
	BTFSS   STATUS,Z
	GOTO    RunAccum
; end of collection period
;
; do linear regression calculation.
;
; Both the Accum and SumXY variables are normalised as if there
; were 1024 samples of DisVal for each of 42 sub periods. The actual
; number of samples depends on the test length, so this may be a left
; or right shift of the values. The formula
;      Delta DisVal = (14*SumXY - 301*Accum)/32
; is applied. This yields a value which is a good statistical
; approximation of the the change of DisVal from beginning to
; end of test, with 16 bits of fraction (i.e. * 2^16)
;
	BSF     STATUS,RP0	; bank 1 for Arithmetic
	MOVLW   SumXY
	CALL    Mpush4
	CALL    normalise
	MOVLW   0x0E ; = 14
	CALL    MpushLit
	CALL    Multiply ; (14 * SumXY)
	MOVLW   Accum
	CALL    Mpush3U
	CALL    normalise
	MOVLW   0x2D ; push 301 ( = 0x012D)
	MOVWF   Mtemp
	MOVLW   0x01
	MOVWF   Mtemp+1
	MOVLW   Mtemp
	CALL    Mpush2U
	CALL    Multiply ; (301*Accum)
	CALL    Msubtract
	MOVLW   0x20 ; = 32
	CALL    MpushLit
	CALL    Mdivide
	MOVLW   DeltaP ; save the calculated DisVal change
	CALL    Mpop4
;	MOVLW	DeltaP
;	MOVWF	FSR
;	MOVLW	0x03 | (('k' & 0x1F)<<3)
;	CALL	TXhex
;
; The target DisVal is 128.5, the calculated end point of the period
; is subtracted from it to give the correction to be applied.
; The value put on the stack is 128.5 * 2^16
;
; Determine average value of DisVal
	MOVLW   Accum       ; total of DisVal
	CALL    Mpush3U
	MOVLW   Counted     ; Number of DisVal samples accumulated
	CALL    Mpush3U
	CALL    MidDiv     ; divide by samples giving mid point number
; The average DisVal is at top of stack. To get an estimated DisVal at
; the end of the accumulation period, half the DeltaP is added to it.
	MOVLW	DeltaP
	CALL	Mpush4
	MOVLW	0xFF ; (-1)
	CALL    Mshift ; divide by 2
	CALL    Madd ; Average Preval + 1/2 DeltaP
; now subtract the calculated value from 128.5 to work out the
; correction needed
	CLRF    Mtemp
	MOVLW   0x80
	MOVWF   Mtemp+1 ; = 128 * 2^8 - equivalent to 0.5 * 2^16
	MOVWF   Mtemp+2 ; = 128 * 2^16
	MOVLW	Mtemp
	CALL	Mpush3U
	CALL    Msubtract
; If the errors are small (less than 6 detector units) for two test
; periods in a row then the test period can be doubled.
	MOVLW	Mtemp ; get the value to test if the period can be increased
	CALL	Mpop4
	MOVLW	Mtemp ; and back on stack for ajustment calculation
	CALL	Mpush4
;	MOVLW	Mtemp
;	MOVWF	FSR
;	MOVLW	0x04 | (('q' & 0x1F)<<3)
;	CALL	TXhex

	BCF     STATUS,RP0
	MOVF    Mtemp+2,W
	BTFSC   Mtemp+3,7   ; get absolute value of error in detector units
	SUBLW   0x00
	SUBLW   0x03        ; less than 3 units ?
	BTFSS   STATUS,C
	GOTO    NotSmall    ; no, don't adjust length
	BSF     STATUS,RP0
	MOVF    DeltaP+2,W
	BTFSC   DeltaP+3,7   ; get absolute value of slope in detector units
	SUBLW   0x00        ; (ignores the fractional part)
	BCF     STATUS,RP0
	SUBLW   0x02         ; less than two units ?
	BTFSS   STATUS,C
	GOTO    NotSmall    ; no, don't adjust length
; a small adjustment, looking for two in a row
	BTFSC   SmallAdj
	GOTO    TwoSmall
	BSF	SmallAdj ; first small adjustment
	GOTO    AdjCalc
TwoSmall:
; Two small adjustments in a row. Increase the measurement period
; if it is less than the limit
	MOVLW   TestLimit
	SUBWF   TstLen,W    ; if TstLen reached the limit, no increase
	BTFSS   STATUS,C
	INCF    TstLen,F
NotSmall:
	BCF	SmallAdj
; top of stack is the residual error. DeltaP is the slope
; applied last period. The change of correction needed is
; (error + DeltaP)
AdjCalc:
	BSF     STATUS,RP0	; bank 1 for Arithmetic
	MOVLW	DeltaP
	CALL	Mpush4
	CALL	Madd
; to get the change to PWM, need to convert the residual correction to
; PWM units using slope
ForceIt: ; entry if the system has gone out of range
	MOVLW   Slope
	CALL    Mpush4
	CALL    MidMul ; multiplies the DisVal change by slope
	BTFSC   STATUS,C
	CALL    ErrEnd ;
; ajust PWM units for longer periods
	DECF    TstLen,W
	SUBLW	0x00 ; needs to be -ve, right shift
	CALL    Mshift
; now add adjustment to PWM
	MOVLW   PWMcalc ; 
	CALL    Mpush3U
	CALL    Madd
	MOVLW   Mtemp
	CALL    Mpop4
        BCF     STATUS,RP0  ; end of arithmetic
;
; diagnostic
	MOVLW	TstLen
	MOVWF	FSR
	MOVLW	0x01 | (('p' & 0x1F)<<3)
	CALL	TXhex
;
	MOVLW   Mtemp
	CALL    LoadPWM
; ready for the next averaging period
	GOTO    FinaLoop

;###########################################
;            END MAINLINE
;###########################################

;------------------- SUBROUTINES -----------------------

CalWait:
;
; Sets the discharge time (DisVal) to the middle of its range.
; Use the reset flag to cause the timing loop to adjust so the
; RB0 interrupts are within 1 instruction time of where DisVal
; expects them. The calling routine is expected to then adjust
; DisVal until it 'locks on' to the RB0 interrupt.
;
	MOVLW   0x80  ; set DisCharge to the mid range (128)
	MOVWF   DisVal
	BSF	ResetCk ; reset counters on change
	CALL    FlagWait     ; one call should do it
	BTFSS	GPStick
	GOTO	CalWait ; if no tick, counters not reset
	BCF	ResetCk ; disable counter resetting
	RETURN

;---------------------------------------------------------

Cal84:
; The code 'zeros in' on the GPS pulse using a series of 3 second binary
; search. It advances or retards the detector by multiples of the LoopC
; time. The intervals are 4,2,1 so the detector can change a maximum of
; +-7 LoopC counts in 3 seconds. The test is run for 96 seconds or 32
; binary searches. The first 4 searches are discarded so the detector
; settles on the pulse. The maximum theoretical tracking range in the time
; is +-32*7 = 224 LoopC counts, but the detector is limited to the range
; +127 -128 and will abort if the limit is reached. In the test time, the
; expected cycle loss or gain should be less than 3 instruction times or 
; 1200nS. This would result in a maximim detector change of around 86 LoopC
; counts (minimum step size of 14nS), which is within the detector limits.
    MOVLW   -4 ; count -4 to zero and start capture at 0
    MOVWF   TstLoop
CalAgain:
; binary search
    MOVLW   0x04
    MOVWF   PreInc
BinInner:
    CALL    JogVal  ; change DisVal
NoJog:
    CALL    FlagWait
    BTFSS   GPStick
    GOTO    NoJog ; missed GPS - shouldn't happen
InnNxt:
    BCF     STATUS,C ; make the next increment smaller
    RRF     PreInc,F
    BTFSS   STATUS,C ; C set, increment is zero, skip
    GOTO    BinInner ; C clear, valid increment
    MOVF    DisVal,W
    INCFSZ  TstLoop,F
    GOTO    $+2
    MOVWF   Mtemp ; save a start value
    MOVLW   0x1C ; = 28
    SUBWF   TstLoop,W ; stop testing at 28 (1/3 of 84)
    BTFSS   STATUS,Z
    GOTO    CalAgain
    MOVF    Mtemp,W ; end of test, retrieve start value
    SUBWF   DisVal,W ; subtract the end value
    RETURN  ; return the difference in W

;---------------------------------------------------------

JogVal:
; Shifts DisVal earlier or later by the amount in PreInc, depending on
; the comparison result. Should be one of only two results 0 or -1. If
; 0 then start the precharge later, makes the transition earlier.
; If -1 then start the precharge earlier, makes the transition later.
; if the calculation overflows (DisVal beyond its limit) an error occurs. 
    MOVF    PreInc,W
    BTFSC   Tcomp,7
    GOTO    JogAdd
; precharge later - subtract
    SUBWF   DisVal,F
    BTFSS   STATUS,C
    CALL    ErrEnd ; gone too far
    GOTO    JogOut
; precharge earlier - add
JogAdd:
    ADDWF   DisVal,F
    BTFSC   STATUS,C
    CALL    ErrEnd ; gone too far
JogOut:
	RETURN

;---------------------------------------------------------

LoadPWM:
; move whatever FSR points to PWMcalc. Flagwait uses PWMcalc and makes
; a temperature adjustmant to get the actual value of PWM
	MOVWF   FSR
	MOVF    INDF,W
	MOVWF   PWMcalc
	INCF    FSR,F
	MOVF    INDF,W
	MOVWF   PWMcalc+1
	INCF    FSR,F
	MOVF    INDF,W
	MOVWF   PWMcalc+2
; diagnostic
	MOVLW	PWMcalc
	MOVWF	FSR
	MOVLW	0x03 | (('v' & 0x1F)<<3)
	GOTO	TXhex
;	RETURN ; if diagnostic is commented, uncomment this

;---------------------------------------------------------

SavLims:
; Called when testing the pulling range of the oscillator. The
; calibration requires that it can be pulled a minimum of
; 1 part in 10^8 high and low of the target frequency.
; This approximated as 1 processor tick per 42 seconds
; and FreqErr returns this as *256 so the minimum value should be
;  => 0x000100 or <= 0xFFFF00 (3 byte signed values)
; If the value is in range, save data in the max or min fields.
;
    BTFSC   Mtemp+2,7   ; sign test
    GOTO    SLneg
    MOVF    Mtemp+1,W   ; test if the msb bytes are both
    IORWF   Mtemp+2,W   ; zero - if so, too small
    BTFSS   STATUS,Z
    GOTO    SLim2 ; not both zero, in range
    RETURN  ; < 0x000100
; test negative
SLneg:
    INCFSZ  Mtemp+2,W
    GOTO    SLim2   ; wasn't 0xFF....
    INCFSZ  Mtemp+1,W
    GOTO    SLim2   ; wasn't 0xFFFF..
    MOVF    Mtemp,W
    BTFSS   STATUS,Z ; accept 0xFFFF00 only
    RETURN  ; > 0xFFFF00
SLim2:
; choose where to store the data
    MOVLW   Vslow
    BTFSS   Mtemp+2,7 ; was it slow or fast
    MOVLW   Vfast
    MOVWF   FSR     ; pointer to destination
; save the PWM value to Vxxx
    MOVF    PWMcalc,W
    MOVWF   INDF
    INCF    FSR,F
    MOVF    PWMcalc+1,W
    MOVWF   INDF
    INCF    FSR,F
    MOVF    PWMcalc+2,W
    MOVWF   INDF
    INCF    FSR,F
; save the ticks difference in Cxxx - entry used while refining Vfast/Vslow
SaveCval:
    MOVF    Mtemp,W
    MOVWF   INDF
    INCF    FSR,F
    MOVF    Mtemp+1,W
    MOVWF   INDF
    INCF    FSR,F
    MOVF    Mtemp+2,W
    MOVWF   INDF
    RETURN

;---------------------------------------------------------

FreqErr:
; Calculates a frequency error for a set value of PWM. The test is run
; for up to 254 seconds, but will exit early if the error is large.
;
; Frequency error is if Tcomp (calculated by FlagWait) is not zero,
; indicating the GPS 1ppS was detected earlier or later than expected
; by Tcomp instruction times.
;
; The frequency error is returned as ticks (Instruction times - 400nS or
; 4 cycles of 10MHz) per 42 seconds at the set duty cycle. This can be a
; fraction so it is returned as 3 bytes:
; Mtemp   - fractions of tick in 1/256ths
; Mtemp+1,2 - whole ticks.
; It is calculated as:
; (difference in ticks*42)/(time of second change - time of first change)
; where difference in ticks is [sign of diff].[magnitude of diff - 1]
; Since difference is a signed number, the result is also signed.
    CLRF    Mtemp   ; stores time of first change
    CLRF    Mtemp+1 ; stores time of last change
    CLRF    Mtemp+2 ; accumulate changes LSB
    CLRF    Mtemp+3 ; accumulate changes MSB
    MOVLW   0x05 ; Delay 5 secs to let voltage settle
    CALL    SecsDly
; test length 254 seconds (avoid recording zero for start or end)
    CLRF    TstLoop
    DECF    TstLoop,F
; delay timer ends, look for next GPS pulse
Ferr2:
    CALL    FlagWait
    BTFSS   GPStick
    GOTO    Ferr2 ; missed GPS tick, ignore
    MOVF    Tcomp,W
    BTFSC   STATUS,Z    ; any tick difference
    GOTO    Ferr3       ; no tick difference, no update
    ADDWF   Mtemp+2,F   ; add to previous recorded difference
    CLRW                ; sign extend Tcomp
    BTFSC   Tcomp,7     ; add or subtract?
    MOVLW   0xFF        ; sign extension
    BTFSC   STATUS,C    ; carry out?
    ADDLW   0x01        ; yes
    ADDWF   Mtemp+3,F
    MOVF    TstLoop,W     ; record time of change
    MOVWF   Mtemp+1
    MOVF    Mtemp,F     ; do we have a start second?
    BTFSC   STATUS,Z
    MOVWF   Mtemp       ; no, save a start time

Ferr3:
; collect info for at least 15 seconds
    MOVLW   0xF0
    SUBWF   TstLoop,W
    BTFSC   STATUS,C
    GOTO    Ferr3A
; if count is more than 15 and difference > 8, exit early
    MOVF    Mtemp+3,F
    BTFSC   STATUS,Z
    GOTO    FerrZ ; MSB is zero, +ve and less than 256, test LSB
    INCFSZ  Mtemp+3,W
    GOTO    Ferr3B ; MSB not 0xFF so > 255 or < -255
FerrZ:
    MOVF    Mtemp+2,W     
    BTFSC   Mtemp+3,7
    SUBLW   0x0
    SUBLW   0x8
    BTFSS   STATUS,C
    GOTO    Ferr3B
Ferr3A:
    DECFSZ  TstLoop,F ; have we reached 254 seconds
    GOTO    Ferr2   ; no, continue the test loop
; calculate the frequency error in processor ticks (400n or 4 cycles
; of 10MHz) per 42 seconds at the set duty cycle.
;
; Special cases exist.
; If the difference didn't change twice while running, Mtemp and Mtemp+1
; are the same. This would result in a divide by zero.
; If the difference was zero, the result is set to zero
; Otherwise the difference should be +-1, set the test length to 255
; and don't change the difference
Ferr3B:
    MOVF    Mtemp+1,W
    SUBWF   Mtemp,F ; Mtemp now time difference first to last
    BTFSS   STATUS,Z ; skip if the duration was zero
    GOTO    Ferr4 ; not zero, treat as a normal case
; special cases
    MOVF    Mtemp+3,W   ; test if any ticks difference saved
    IORWF   Mtemp+2,W
    BTFSC   STATUS,Z    ; if not, no need to go further
    RETURN      ; Mtemp, Mtemp+1 should also be zero so return zero
; otherwise we had one tick change in 254 seconds
    MOVLW   0xFF    ; ticks difference not zero (should be +-1)
    MOVWF   Mtemp   ; so set the time difference to 255 seconds
    GOTO    Ferr5   ; and skip over the ticks adjust
Ferr4:
; Reduce tick difference by 1 in magnitude
    CLRF    Mtemp+1 ; make this the sign extension
    MOVLW   0x01    ; assume negative and adding 1
    BTFSC   Mtemp+3,7 ; test the sign
    GOTO    $+3
    MOVLW   0xFF    ; it's +ve - reduce by 1
    MOVWF   Mtemp+1 ; change the extension
    ADDWF   Mtemp+2,F
    MOVF    Mtemp+1,W
    BTFSC   STATUS,C
    INCF    Mtemp+1,W
    ADDWF   Mtemp+3,F
Ferr5:
; calculate (ticks difference)/(time difference)
    CLRF    Mtemp+1     ; this is msb of Mtemp[0-1] also lsb of Mtemp[1-3]
    BSF     STATUS,RP0  ; Bank 1 for arithmetic
    MOVLW   Mtemp+1     ; push the ticks difference (Mtemp+1 thru +3)
    CALL    Mpush3       ; as a 3 byte number - effectively * 256
    MOVLW   0x2A ; 42
    CALL    MpushLit
    CALL    Multiply       ; and multiply by 42
    MOVLW   Mtemp       ; push the time difference
    CALL    Mpush2
    CALL    Mdivide
    MOVLW   Mtemp   ; pop result back into Mtemp
    CALL    Mpop3
    BCF     STATUS,RP0  ; end of arithmetic
    RETURN

;---------------------------------------------------------

SecsDly:
;
; The PWM output goes through a filter so it takes time for the control
; voltage to reflect a change of PWM. This routine is usually called as
; a delay waiting for the control voltage to settle on the new setting.
;
	MOVWF   Seconds
SDloop:
	MOVLW   0xF6 ; should flash LED2 on 1/2 second off 1/2 second
	MOVWF   Led2of
	MOVLW   0xE2 ; d'-30'
	MOVWF   Led2on
	CALL    FlagWait ; wait for next change
	DECFSZ  Seconds, F
	GOTO    SDloop
	RETURN

;---------------------------------------------------------

FlagWait:
;
; Once GPS 1ppS is enabled, FlagWait is called waiting for the next
; end of each second. Before waiting, it sets up the charge/discharge
; timing if needed. It also applies any temperature adjustment to the
; Control voltage.
; While looping, it polls the LoopC value and turns LEDs on and off
; as required.
; When the second is up, the time of GPS interrupt is compared with
; the target value and the difference in instruction times is returned
; as a signed 1 byte value in Tcomp. The value returned is normally a
; real value in the range +-100, but if there is a bigger difference it
; is returned as a false value + or - 101. When the control voltage is
; near the correct value the only expected values are 0x00 or 0xFF (-1)
;
	BCF	LoopEnd ; wait til this is set
	BCF	GPStick 
	BTFSC   VRon
	GOTO    RunVR ; using fixed voltage - skip charge/discharge
;
; Set the timing for charging and discharging the detector capacitor.
; The charge length is derived from previously timed charging.
; end of precharge is set notionally 5mS (50 TMR2 interrupts) before
; start of discharge. This may vary in practice as the actual charge
; time can change slightly. For this period the charge should not change
; as both ports are high impedence. The discharge starts a differing number
; of LoopC counts before the expected arrival time of the next GPS pulse.
; This adjusts the delay between the arrival of the GPS pulse and its
; detection.
; First calculate a smoothed value for charging:
; Subtract 1/8th of ChgLen from ChgLen then add in the latest TMR1 value.
; 1/256th of ChgLen is approx charge time in 100uS intervals.
; Don't worry about C moving into ChgWork. They are ignored.
	RRF	ChgLen+2,W ; three shifts to divide by 8
	MOVWF	ChgWork
	RRF	ChgLen+1,W
	MOVWF	ChgAvg+1
	RRF	ChgLen,W
	MOVWF	ChgAvg
	RRF	ChgWork,F
	RRF	ChgAvg+1,F
	RRF	ChgAvg,F
	RRF	ChgWork,F
	RRF	ChgAvg+1,F
	RRF	ChgAvg,F
	MOVF	ChgAvg,W
	SUBWF	ChgLen,F
	MOVF	ChgAvg+1,W
	BTFSS   STATUS,C
	INCFSZ	ChgAvg+1,W
	SUBWF	ChgLen+1,F
; ChgWork ignored (average of 8 two byte numbers = a two byte number)
	BTFSS   STATUS,C
	DECF	ChgLen+2,F
; now add in the latest TMR1
	MOVF	TMR1L,W
	ADDWF	ChgLen,F
	MOVF	TMR1H,W
	BTFSC	STATUS,C
	INCFSZ	TMR1H,W
	ADDWF	ChgLen+1,F
	BTFSC	STATUS,C
	INCF	ChgLen+2,F
;
; diagnostic
	MOVLW	TMR1L
	MOVWF	FSR
	MOVLW	0x02 | (('t' & 0x1F)<<3)
	CALL	TXhex
; The target for starting discharge is: (Expected LoopC value
; when the last GPS pulse should have arrived) - 2 - adjustment
	MOVLW	RB0Lo - DisOff ; apply the fixed offset
	MOVWF	DchTime
	MOVLW	RB0Hi
	MOVWF   DchTime+1
; subtract the current value of DisVal (the adjustment)
	MOVF    DisVal,W  ; DisVal
	SUBWF   DchTime,F    ; subtract it
	BTFSS   STATUS,C
	DECF    DchTime+1,F
; DchTime now where discharge should start
; calculate time to start charging.
	MOVLW   ChgOff    ; subtract the nominal gap
	SUBWF   DchTime,W ; between charge and discharge
	MOVWF	ChTime
	MOVF    DchTime+1,W
	BTFSS   STATUS,C
	DECF    DchTime+1,W
; Keep the value for ChTime+1 in FSR until everything else is set up.
; This is so the TMR2 interrupt can't start a charge/discharge cycle
; until set up is completed.
	MOVWF	FSR
; now subtract the charge time. 
; ChgLen+1,2 approx 100uS intervals for charging
	MOVF    ChgLen+1,W    ; the start is (end - charge length)
	SUBWF   ChTime,F
	MOVF    ChgLen+2,W
	BTFSS   STATUS,C
	INCF    ChgLen+2,W
	SUBWF   FSR,W ; temporary store for ChTime+1
	MOVWF	ChTime+1
	GOTO	FlagTadj
RunVR:
    	BCF	INTCON,INTF ; clear RB0 interrupt (if any)
	BSF	INTCON,INTE ; and allow the next one
FlagTadj:
; temperature adjust PWMcalc to give a final PWM
; as a trial try ChgAvg*3/8
	BSF     STATUS,RP0	; bank 1 for Arithmetic
	MOVLW	PWMcalc		; add adjustment to this later
	CALL	Mpush3U
;	MOVLW	ChgAvg		; multiply ChgAvg by 3
;	CALL	Mpush2U
;	MOVLW	0x03
;	CALL	MpushLit
;	CALL	Multiply
;	MOVLW	-3		; divide by 8
;	CALL	Mshift
;	CALL	Madd		; add adjustment to PWMcalc
	MOVLW   PWMwork		; get result
	CALL    Mpop3
	BCF     STATUS,RP0	; bank 0 after arithmetic
	MOVF	PWMwork,W	    ; copy PWMwork to PWM
	BCF     INTCON, GIE ; disable interrupts
	BTFSC	INTCON, GIE ; recommended in doco
	GOTO	$-2
	MOVWF	PWM
	MOVF	PWMwork+1,W
	MOVWF	PWM+1
	MOVF	PWMwork+2,W
	MOVWF	PWM+2
	BSF     INTCON, GIE     ; enable interrupts
; debug
;	MOVLW	PWM
;	MOVWF	FSR
;	MOVLW	0x02 | (('q' & 0x1F)<<3)
;	CALL	TXhex
; **** NO TEMPERATURE ADJUST ****
;	MOVF	PWMcalc,W
;	BCF     INTCON, GIE ; disable interrupts
;	BTFSC	INTCON, GIE ; recommended in doco
;	GOTO	$-2
;	MOVWF	PWM
;	MOVF	PWMcalc+1,W
;	MOVWF	PWM+1
;	MOVF	PWMcalc+2,W
;	MOVWF	PWM+2
;	BSF     INTCON, GIE     ; enable interrupts

FlagLoop:
	BTFSC   LoopEnd ; check other things until end of second
	GOTO    FlagSet
;
; check LED counters to see if LEDs are to be turned on or off
; LED logic is inverted, low out turns LED on, high turns LED off
;
; LED1
	MOVF    Led1on,W
	SUBWF   LoopC+1,W
	BTFSC   STATUS,Z ; Match hi bytes
	BCF     LED1 ; turn on
;
	MOVF    Led1of,W
	SUBWF   LoopC+1,W
	BTFSC   STATUS,Z ; Match hi bytes
	BSF     LED1 ; turn off
; LED2
	MOVF    Led2on,W
	SUBWF   LoopC+1,W
	BTFSC   STATUS,Z ; Match hi bytes
	BCF     LED2 ; turn on
;
	MOVF    Led2of,W
	SUBWF   LoopC+1,W
	BTFSC   STATUS,Z ; Match hi bytes
	BSF     LED2 ; turn off
; ** DEBUG check timer
	
	GOTO	FlagLoop

FlagSet:
; end of second.
; If a GPS pulse detected, set up to flash LED1
	BTFSS	GPStick
	RETURN
	MOVLW	0xEB ; d'-21'
	MOVWF	Led1on
; Compares the time of an RB0 event to the expected value. This may
; be used by the mainline to determine differences between the GPS
; and the oscillator. Contributing factors
; to a difference are (1) GPS signal variations and (2) oscillator
; frequency errors. A difference of one tick between events is a
; difference of 0-400nS, two ticks is 400-800nS etc. A GPS error
; of 100 meter between readings (most unlikely) is equivalent to a
; time error of 5 parts in 10^6 - 5uS - 12-13 ticks. A bad crystal
; oscillator could be off by 1 part in 10^5 (also unlikely) or
; 25-26 ticks in a second. For maximum flexibility allow up to
; 100 ticks, anything bigger than that is almost guaranteed to be
; a problem so set it as +-101

; only differences of TMR2 readings are expected. If the LoopC
; counters were not the set RB0Loop value, something is very wrong
	MOVF	Ticks+1,W
	XORLW	RB0Lo
	BTFSS	STATUS,Z ; should be zero
	GOTO    SetStat
	MOVF	Ticks+2,W
	XORLW	RB0Hi
	BTFSS	STATUS,Z ; should be zero
	GOTO    SetStat
	MOVLW   TargVal ; compare with target value. Ordinarily this is
	SUBWF   Ticks,W ; the only difference detected, so
	MOVWF   Tcomp   ; store it
	BTFSS   STATUS,C ; get magnitude
	SUBLW   0x00   ; converts -ve to +ve
; now test if it was within range 0 to +-100
	ADDLW   0x9B ; in range should result in 155 to 255 and C clear
	BTFSS   STATUS,C
	GOTO	CompHex
	MOVF	Tcomp,W
SetStat:
; Expected Ticks is out of range. Set a value +-101 in Tcomp to indicate
; the sign of the error.
	ANDLW   0x7F ; isolate the sign bit, sets or clears Z
; comparison fail. Assume a +ve difference first
	MOVLW   0x65        ; assume +ve, value is +101
	BTFSS   STATUS,Z    ; Z set = +ve, Z clear = -ve
	SUBLW   0x00        ; -ve, value is -101
	MOVWF   Tcomp       ; overwrite Tcomp with the error value
CompHex:
;	MOVLW	Tcomp
;	MOVWF	FSR
;	MOVLW	0x01 | (('e' & 0x1F)<<3)
;	GOTO	TXhex
    RETURN
	
;---------------------------------------------------------
	
Stage:
	INCF    ErrVal,F ; 
; diagnostic
	MOVLW	ErrVal
	MOVWF	FSR
	MOVLW	0x01 | (('s' & 0x1F)<<3)
	GOTO	TXhex
	
;---------------------------------------------------------

normalise:
; used to make variables for linear regression look like there
; were 1024 samples in 42 time periods
    MOVF   TstLen,W
    SUBLW   0x0A ; = 10
    GOTO    Mshift

;---------------------------------------------------------

; ##################### ERROR ROUTINE ##################
; This error routine is called with a value in ErrVal. It
; should display this as a 4 bit binary on the LEDs
; LED1 is a 1 bit, LED2 is a 0 bit
; so Error 5 would show as:
; LED2 flash, LED1 flash, LED2 flash, LED1 flash, every few seconds
; In software, there is no path to RETURN. In debugging, the
; CALL rather than GOTO is useful as the emulator can be
; forced to RETURN, so it is know exactly where in the code
; the error was created.
ErrEnd:
; Not sure what went wrong so set up some things from scratch
; stop any interrupts
	BCF     INTCON, GIE
	BTFSC	INTCON, GIE ; recommended in doco
	GOTO	$-2
	CLRF    STATUS ; make sure we are in Bank 0
	BSF	LED1 ; set the LED pins high (LEDS off)
	BSF	LED2
	BSF     STATUS, RP0 ; Select Bank1
	MOVLW   0xFC        ; Value used to initialize PORTA data direction
	MOVWF   TRISA       ; Set RA0,1 as output
; Set up and start Timer 1
	BCF     STATUS, RP0 ; Select Bank0
	MOVLW   0x31
	MOVWF   T1CON
	CALL	ErrWait
ErrOutr:
	SWAPF   ErrVal,W
	MOVWF   Mtemp
	MOVLW   0x04
	MOVWF   Mtemp+1
ErrFlash:
	CALL    ErrWait
	RLF     Mtemp,F ; get a bit from the error code
	BTFSS   STATUS,C ; decide if it is LED1 or LED2 to turn on
	BCF     LED1
	BTFSC   STATUS,C
	BCF     LED2
	CALL    ErrWait
	CALL    ErrWait
	BSF     LED2 ; turn off both LEDs
	BSF     LED1
	DECFSZ  Mtemp+1,F ; are there more bits to show
	GOTO    ErrFlash
	MOVLW   0x06
	MOVWF   Mtemp+1
ErrDly:
	CALL    ErrWait
	DECFSZ  Mtemp+1,F ; are there more bits to show
	GOTO    ErrDly
	GOTO    ErrOutr ; and do it all again
	RETURN  ; used when debugging
ErrWait:
; wait for TMR1 to overflow - about 200mS
	BTFSS   PIR1,TMR1IF
	GOTO    ErrWait
	BCF	PIR1,TMR1IF
	RETURN

	END
